// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

import 'chrome://resources/mwc/@material/web/button/filled-button.js';
import 'chrome://resources/mwc/@material/web/icon/icon.js';
import 'chrome://resources/mwc/@material/web/iconbutton/icon-button.js';
import '../components/cra/cra-icon.js';

import {createRef, css, html, ref} from 'chrome://resources/mwc/lit/index.js';

import {
  usePlatformHandler,
  useRecordingDataManager,
} from '../core/lit/context.js';
import {ModelResponse} from '../core/on_device_model/types.js';
import {ReactiveLitElement} from '../core/reactive/lit.js';
import {signal} from '../core/reactive/signal.js';
import {LanguageCode} from '../core/soda/language_info.js';
import {TestHelper} from '../core/test_helper.js';

/**
 * Dev page of Recorder App.
 */
export class DevPage extends ReactiveLitElement {
  static override styles = css`
    :host {
      display: block;
      height: 100%;
      position: relative;
      width: 100%;
    }

    .container {
      display: flex;
      flex-direction: column;
      gap: 8px;
      height: 100%;
      width: 100%;
    }

    .section {
      align-items: center;
      background: var(--cros-sys-surface1);
      display: flex;
      flex-direction: column;
      padding: 8px;
    }

    md-icon-button {
      align-self: start;
    }
  `;

  private readonly platformHandler = usePlatformHandler();

  readonly recordingDataManager = useRecordingDataManager();

  private readonly textareaRef = createRef<HTMLTextAreaElement>();

  private readonly inputRef = createRef<HTMLInputElement>();

  private readonly importRecordingDataResult = signal<string>('');

  /**
   * Contains model response of the suggested titles.
   */
  private readonly titles = signal<ModelResponse<string[]>|null>(null);

  private async onSuggestTitleClick() {
    const value = this.textareaRef.value?.value;
    if (value === undefined) {
      return;
    }
    // TODO(shik): Add loading state.
    // TODO(hsuanling): Supports different language
    this.titles.value =
      await this.platformHandler.titleSuggestionModelLoader.loadAndExecute(
        value,
        LanguageCode.EN_US,
      );
  }

  private async onClearClicked() {
    await this.recordingDataManager.clear();
  }

  private async onRecordingDataImport() {
    const files = this.inputRef.value?.files ?? null;
    this.importRecordingDataResult.value = 'Importing…';

    if (files === null) {
      this.importRecordingDataResult.value = '';
      return;
    }

    const importErrors: string[] = [];
    for (const file of files) {
      try {
        const fileContent = await file.text();
        await TestHelper.importRecordings(JSON.parse(fileContent));
      } catch (err) {
        importErrors.push(`Failed to import ${file.name}: ${err}`);
      }
    }
    if (importErrors.length > 0) {
      this.importRecordingDataResult.value = importErrors.join('\n');
    } else {
      this.importRecordingDataResult.value = 'Imported successfully';
    }
  }

  override render(): RenderResult {
    // TODO(shik): Make this prettier.
    return html`
      <div class="container">
        <md-icon-button href="/">
          <cra-icon name="arrow_back"></cra-icon>
        </md-icon-button>
        <div class="section">
          <md-filled-button @click=${this.onClearClicked}
            >Clear OPFS</md-filled-button
          >
        </div>
        <div class="section">
          <textarea ${ref(this.textareaRef)} rows=${5}></textarea>
          <pre>Titles: ${JSON.stringify(this.titles.value, null, 2)}</pre>
          <md-filled-button @click=${this.onSuggestTitleClick}>
            Suggest Title
          </md-filled-button>
        </div>
        <div class="section">
          <pre>Select RecordingData to import</pre>
          <input type="file" accept=".json" multiple
            ${ref(this.inputRef)} @change=${this.onRecordingDataImport}>
          <p>${this.importRecordingDataResult.value}</p>
        </div>
        ${this.platformHandler.renderDevUi()}
      </div>
    `;
  }
}

window.customElements.define('dev-page', DevPage);

declare global {
  interface HTMLElementTagNameMap {
    'dev-page': DevPage;
  }
}
